   // Chat Widget JavaScript with Firebase integration
        window.CustomChatWidget = {
            isOpen: false,
            isLoading: false,
            messages: [],
            sessionInitialized: false,
            unsubscribe: null,

            // DOM elements
            chatButton: null,
            chatWindow: null,
            closeButton: null,
            messagesArea: null,
            messageInput: null,
            sendButton: null,

            // Initialize
            init: function() {
                this.chatButton = document.getElementById('custom-chat-widget-button');
                this.chatWindow = document.getElementById('custom-chat-widget-window');
                this.closeButton = document.getElementById('custom-chat-widget-close-button');
                this.messagesArea = document.getElementById('custom-chat-widget-messages-area');
                this.messageInput = document.getElementById('custom-chat-widget-message-input');
                this.sendButton = document.getElementById('custom-chat-widget-send-button');

                // Event listeners
                this.chatButton.addEventListener('click', () => this.openChat());
                this.closeButton.addEventListener('click', () => this.closeChat());
                this.sendButton.addEventListener('click', () => this.sendMessage());
                this.messageInput.addEventListener('keypress', (e) => this.handleKeyPress(e));

                // Initialize Firebase chat session
                this.initializeChatSession();
            },

            openChat: function() {
                this.isOpen = true;
                this.chatButton.classList.add('hidden');
                this.chatWindow.classList.remove('hidden');
                this.messageInput.focus();
            },

            closeChat: function() {
                this.isOpen = false;
                this.chatButton.classList.remove('hidden');
                this.chatWindow.classList.add('hidden');
            },

            handleKeyPress: function(e) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            },

            getSessionId: async function() {
                let sessionId = localStorage.getItem('custom_chat_widget_session_id');
                if (!sessionId) {
                    try {
                        // Create anonymous user if Firebase Auth is available
                        if (window.firebaseAuth && window.firebaseSignInAnonymously) {
                            const userCredential = await window.firebaseSignInAnonymously(window.firebaseAuth);
                            sessionId = userCredential.user.uid;
                        } else {
                            // Fallback to random ID if Firebase Auth is not available
                            sessionId = 'custom_session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
                        }
                        localStorage.setItem('custom_chat_widget_session_id', sessionId);
                    } catch (error) {
                        console.error('Error creating anonymous user:', error);
                        // Fallback to random ID if anonymous sign in fails
                        sessionId = 'custom_session_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
                        localStorage.setItem('custom_chat_widget_session_id', sessionId);
                    }
                }
                return sessionId;
            },

            // Initialize Firebase chat session
            initializeChatSession: async function() {
                if (this.sessionInitialized) return;

                try {
                    const sessionId = await this.getSessionId();

                    // Check if session exists, if not create it
                    const existingSession = await this.getChatSession(sessionId);
                    if (!existingSession) {
                        await this.createChatSession(sessionId);
                        // Add initial bot message
                        await this.addMessageToFirebase(sessionId, {
                            text: "Hello! How can I help you today?",
                            isBot: true
                        });
                    }

                    // Listen to messages in real-time
                    this.listenToMessages(sessionId);
                    this.sessionInitialized = true;
                } catch (error) {
                    console.error('Error initializing chat session:', error);
                    // Fallback to local message if Firebase fails
                    this.addMessageToUI("Hello! How can I help you today?", true);
                    this.sessionInitialized = true;
                }
            },

            // Create chat session in Firebase
            createChatSession: async function(sessionId) {
                const chatSessionRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}`);
                const sessionData = {
                    sessionId,
                    createdAt: Date.now(),
                    lastActivity: Date.now(),
                    status: 'active',
                    userInfo: {
                        userAgent: navigator.userAgent,
                        timestamp: Date.now(),
                        url: window.location.href
                    },
                    messages: {}
                };

                await window.firebaseSet(chatSessionRef, sessionData);
                return sessionData;
            },

            // Add message to Firebase
            addMessageToFirebase: async function(sessionId, messageData) {
                const messagesRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}/messages`);
                const messageRef = window.firebasePush(messagesRef);

                const message = {
                    id: messageRef.key,
                    text: messageData.text,
                    isBot: messageData.isBot || false,
                    timestamp: Date.now(),
                    createdAt: new Date().toISOString()
                };

                await window.firebaseSet(messageRef, message);

                // Update last activity
                const lastActivityRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}/lastActivity`);
                await window.firebaseSet(lastActivityRef, Date.now());

                return message;
            },

            // Get chat session from Firebase
            getChatSession: function(sessionId) {
                return new Promise((resolve, reject) => {
                    const sessionRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}`);
                    window.firebaseOnValue(sessionRef, (snapshot) => {
                        if (snapshot.exists()) {
                            resolve(snapshot.val());
                        } else {
                            resolve(null);
                        }
                    }, reject);
                });
            },

            // Listen to messages in real-time
            listenToMessages: function(sessionId) {
                const messagesRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}/messages`);
                const messagesQuery = window.firebaseQuery(messagesRef, window.firebaseOrderByChild('timestamp'));

                window.firebaseOnValue(messagesQuery, (snapshot) => {
                    this.messagesArea.innerHTML = ''; // Clear existing messages

                    if (snapshot.exists()) {
                        const messages = [];
                        snapshot.forEach((childSnapshot) => {
                            messages.push({
                                id: childSnapshot.key,
                                ...childSnapshot.val()
                            });
                        });

                        // Display messages with date separators
                        this.renderMessagesWithDates(messages);
                    }
                });
            },

            // Render messages with date separators
            renderMessagesWithDates: function(messages) {
                const messagesWithDates = this.groupMessagesByDate(messages);

                messagesWithDates.forEach(item => {
                    if (item.type === 'date') {
                        this.addDateSeparator(item.label);
                    } else {
                        this.addMessageToUI(item.message.text, item.message.isBot, false);
                    }
                });

                // Scroll to bottom
                this.messagesArea.scrollTop = this.messagesArea.scrollHeight;
            },

            // Add date separator
            addDateSeparator: function(label) {
                const separatorDiv = document.createElement('div');
                separatorDiv.className = 'flex justify-center my-4';
                separatorDiv.innerHTML = `
                    <div class="bg-gray-100 text-gray-600 text-xs px-3 py-1 rounded-full">
                        ${label}
                    </div>
                `;
                this.messagesArea.appendChild(separatorDiv);
            },

            // Safely render markdown for bot messages and plain text for user messages
            renderMessageContent: function(containerEl, text, isBot) {
                const textEl = document.createElement('div');
                textEl.className = 'custom-chat-widget-message-text text-sm chat-markdown';
                // Use pre-wrap for user text to preserve their manual line breaks
                // and normal whitespace for bot markdown (HTML already controls spacing)
                textEl.style.whiteSpace = isBot ? 'normal' : 'pre-wrap';

                if (isBot && window.marked && window.DOMPurify) {
                    try {
                        // Normalize input: convert <br> tags to newlines and unify line endings
                        let src = String(text || '');
                        src = src.replace(/\r\n/g, '\n');
                        // Handle literal <br> tags (both real tags and escaped &lt;br&gt;)
                        src = src.replace(/<br\s*\/?>(\s*)/gi, '\n');
                        src = src.replace(/&lt;br\s*\/?&gt;(\s*)/gi, '\n');
                        // Collapse 3+ blank lines to at most 1 blank line between paragraphs
                        src = src.replace(/\n{3,}/g, '\n\n');

                        // Configure marked for soft line breaks and GitHub-flavored markdown
                        window.marked.setOptions({
                            breaks: true,
                            gfm: true,
                            headerIds: false,
                            mangle: false
                        });
                        const rawHtml = window.marked.parse(src);
                        const cleanHtml = window.DOMPurify.sanitize(rawHtml, { USE_PROFILES: { html: true } });
                        textEl.innerHTML = cleanHtml;

                        // Ensure links open in a new tab and are safe
                        textEl.querySelectorAll('a').forEach(a => {
                            a.setAttribute('target', '_blank');
                            a.setAttribute('rel', 'noopener noreferrer nofollow');
                        });
                    } catch (e) {
                        console.error('Markdown render error:', e);
                        textEl.textContent = text || '';
                    }
                } else {
                    // For user messages or if libs are unavailable, render as plain text
                    textEl.textContent = text || '';
                }

                containerEl.appendChild(textEl);
            },

            // Add message to UI
            addMessageToUI: function(text, isBot = false, scroll = true) {
                const messageDiv = document.createElement('div');
                messageDiv.className = `custom-chat-widget-message-container flex ${isBot ? 'justify-start' : 'justify-end'}`;

                const timestamp = new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });

                // Outer bubble
                const bubble = document.createElement('div');
                bubble.className = `custom-chat-widget-message max-w-xs lg:max-w-md px-4 py-2 rounded-lg ${isBot ? 'bg-white text-gray-800' : 'text-gray-800'}`;
                if (!isBot) {
                    bubble.style.backgroundColor = '#E3DEEE';
                }

                // Message body
                this.renderMessageContent(bubble, text, isBot);

                // Timestamp
                const timeEl = document.createElement('p');
                timeEl.className = 'custom-chat-widget-message-time text-xs opacity-70 mt-1';
                timeEl.textContent = timestamp;
                bubble.appendChild(timeEl);

                messageDiv.appendChild(bubble);

                this.messagesArea.appendChild(messageDiv);
                if (scroll) {
                    this.messagesArea.scrollTop = this.messagesArea.scrollHeight;
                }
            },

            addMessage: function(text, isBot = false) {
                // Delegate to addMessageToUI to keep behavior consistent
                this.addMessageToUI(text, isBot, true);
            },

            showLoading: function() {
                const loadingDiv = document.createElement('div');
                loadingDiv.id = 'custom-chat-widget-loading-message';
                loadingDiv.className = 'custom-chat-widget-loading-container flex justify-start';
                loadingDiv.innerHTML = `
                    <div class="custom-chat-widget-loading-message bg-gray-200 text-gray-800 px-4 py-2 rounded-lg flex items-center space-x-2">
                        <div class="custom-chat-widget-loading-spinner w-4 h-4 border-2 border-gray-600 border-t-transparent rounded-full chat-widget-spin"></div>
                        <span class="custom-chat-widget-loading-text text-sm">Typing...</span>
                    </div>
                `;
                this.messagesArea.appendChild(loadingDiv);
                this.messagesArea.scrollTop = this.messagesArea.scrollHeight;
            },

            hideLoading: function() {
                const loadingMessage = document.getElementById('custom-chat-widget-loading-message');
                if (loadingMessage) {
                    loadingMessage.remove();
                }
            },

            sendMessage: async function() {
                const message = this.messageInput.value.trim();
                if (!message || this.isLoading || !this.sessionInitialized) return;

                this.messageInput.value = '';
                this.isLoading = true;
                this.sendButton.disabled = true;
                this.messageInput.disabled = true;

                // Show loading
                this.showLoading();

                try {
                    const sessionId = await this.getSessionId();

                    // Save user message to Firebase
                    await this.addMessageToFirebase(sessionId, {
                        text: message,
                        isBot: false
                    });

                    // Send to server proxy (POST)
                    const payload = {
                        payload: {
                            sessionId: sessionId,
                            action: 'sendMessage',
                            chatInput: message
                        }
                    };

                    const response = await fetch(`${window.CHAT_API_CONFIG.apiBaseUrl}/proxy.php?action=chats`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify(payload)
                    });

                    const data = await response.json();

                    // Hide loading
                    this.hideLoading();

                    // Save bot response to Firebase
                    await this.addMessageToFirebase(sessionId, {
                        text: data.reply || 'Sorry, I encountered an error. Please try again.',
                        isBot: true
                    });

                } catch (error) {
                    console.error('Error sending message:', error);
                    this.hideLoading();

                    // Save error message to Firebase
                    try {
                        const sessionId = await this.getSessionId();
                        await this.addMessageToFirebase(sessionId, {
                            text: 'Sorry, I encountered an error. Please try again.',
                            isBot: true
                        });
                    } catch (firebaseError) {
                        console.error('Error saving error message to Firebase:', firebaseError);
                        // Fallback to UI only
                        this.addMessageToUI('Sorry, I encountered an error. Please try again.', true);
                    }
                } finally {
                    this.isLoading = false;
                    this.sendButton.disabled = false;
                    this.messageInput.disabled = false;
                    this.messageInput.focus();
                }
            },

            // Group messages by date with WhatsApp-style separators
            groupMessagesByDate: function(messages) {
                if (!messages || messages.length === 0) return [];

                const result = [];
                let currentDate = null;

                messages.forEach(message => {
                    const messageDate = this.getDateString(message.timestamp);

                    if (currentDate !== messageDate) {
                        currentDate = messageDate;
                        result.push({
                            type: 'date',
                            label: this.formatDateSeparator(message.timestamp)
                        });
                    }

                    result.push({
                        type: 'message',
                        message: message
                    });
                });

                return result;
            },

            // Get date string for comparison (YYYY-MM-DD)
            getDateString: function(timestamp) {
                const date = new Date(timestamp);
                return date.toISOString().split('T')[0];
            },

            // Format date separator like WhatsApp
            formatDateSeparator: function(timestamp) {
                const messageDate = new Date(timestamp);
                const today = new Date();
                const yesterday = new Date(today);
                yesterday.setDate(yesterday.getDate() - 1);

                // Reset time to compare dates only
                const messageDateOnly = new Date(messageDate.getFullYear(), messageDate.getMonth(), messageDate.getDate());
                const todayOnly = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                const yesterdayOnly = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate());

                if (messageDateOnly.getTime() === todayOnly.getTime()) {
                    return 'Today';
                } else if (messageDateOnly.getTime() === yesterdayOnly.getTime()) {
                    return 'Yesterday';
                } else {
                    return messageDate.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                    });
                }
            }
        };

        // Initialize the chat widget when DOM is loaded
        document.addEventListener('DOMContentLoaded', function() {
            // Wait for Firebase to be available
            const checkFirebase = setInterval(() => {
                if (window.firebaseDatabase) {
                    clearInterval(checkFirebase);
                    window.CustomChatWidget.init();
                }
            }, 100);
        });