<?php
header('Content-Type: application/json; charset=utf-8');

// Try to include secrets from outside web root; fallback to local sample for dev
$ALLOWED_ORIGINS = '*';
$INTERNAL_API_KEY = '';
$WEBHOOK_CHAT_URL = '';
$WEBHOOK_KB_URL = '';

// Attempt to load real secrets (adjust path on cPanel)
$externalSecrets = '/home/monazem/public_html/secrets-yicen.php';
if (@is_readable($externalSecrets)) {
  require $externalSecrets;
  // Expect constants defined: ALLOWED_ORIGINS, INTERNAL_API_KEY, WEBHOOK_CHAT_URL, WEBHOOK_KB_URL
  $ALLOWED_ORIGINS = defined('ALLOWED_ORIGINS') ? ALLOWED_ORIGINS : $ALLOWED_ORIGINS;
  $INTERNAL_API_KEY = defined('INTERNAL_API_KEY') ? INTERNAL_API_KEY : $INTERNAL_API_KEY;
  $WEBHOOK_CHAT_URL = defined('WEBHOOK_CHAT_URL') ? WEBHOOK_CHAT_URL : $WEBHOOK_CHAT_URL;
  $WEBHOOK_KB_URL = defined('WEBHOOK_KB_URL') ? WEBHOOK_KB_URL : $WEBHOOK_KB_URL;
} elseif (@is_readable(__DIR__ . '/secrets.dev.php')) {
  require __DIR__ . '/secrets.dev.php'; // dev-only fallback (not for production)
  $ALLOWED_ORIGINS = defined('ALLOWED_ORIGINS') ? ALLOWED_ORIGINS : $ALLOWED_ORIGINS;
  $INTERNAL_API_KEY = defined('INTERNAL_API_KEY') ? INTERNAL_API_KEY : $INTERNAL_API_KEY;
  $WEBHOOK_CHAT_URL = defined('WEBHOOK_CHAT_URL') ? WEBHOOK_CHAT_URL : $WEBHOOK_CHAT_URL;
  $WEBHOOK_KB_URL = defined('WEBHOOK_KB_URL') ? WEBHOOK_KB_URL : $WEBHOOK_KB_URL;
}

// Basic CORS
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$allowed = array_map('trim', explode(',', $ALLOWED_ORIGINS));
if ($origin && ($ALLOWED_ORIGINS === '*' || in_array($origin, $allowed, true))) {
  header("Access-Control-Allow-Origin: $origin");
  header('Vary: Origin');
}
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); echo json_encode(['error' => 'Only POST allowed']); exit; }

$action = $_GET['action'] ?? '';
$input = json_decode(file_get_contents('php://input'), true) ?? [];
$payload = $input['payload'] ?? [];

$map = [
  'chats' => $WEBHOOK_CHAT_URL,
  'knowledgebase' => $WEBHOOK_KB_URL,
];

$targetUrl = $map[$action] ?? '';

// If no target configured, return a safe stubbed response (for dev/testing)
if (!$targetUrl) {
  if ($action === 'chats') {
    $chatInput = isset($payload['chatInput']) ? (string)$payload['chatInput'] : '';
    echo json_encode(['reply' => 'Thanks! (stubbed response) ' . ($chatInput ? 'You said: ' . substr($chatInput, 0, 120) : '')]);
    exit;
  }
  if ($action === 'knowledgebase') {
    echo json_encode(['status' => 'ok', 'message' => 'Stubbed KB request']);
    exit;
  }
  http_response_code(400); echo json_encode(['error' => 'Unknown action or not configured']); exit;
}

// Forward to upstream webhook (method depends on action)
$headers = [
  'Content-Type: application/json',
];
if (!empty($INTERNAL_API_KEY)) {
  $headers[] = 'X-Internal-Api-Key: ' . $INTERNAL_API_KEY;
}

$ch = null;
if ($action === 'chats') {
  // Original client used GET with query params
  $qs = http_build_query($payload);
  $url = $targetUrl . (strpos($targetUrl, '?') !== false ? '&' : '?') . $qs;
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPGET => true,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_TIMEOUT => 15,
  ]);
} else {
  // Default to POST JSON
  $ch = curl_init($targetUrl);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_TIMEOUT => 15,
  ]);
}

$response = curl_exec($ch);
$status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$err = curl_error($ch);
curl_close($ch);

if ($err) { http_response_code(502); echo json_encode(['error' => 'Upstream error', 'detail' => $err]); exit; }
http_response_code($status ?: 200);
echo $response;
